// ==============================================================
// Nieuwswijs Chrome Extension - Background Service Worker
// ==============================================================
// Manages overlay state across tab switches.
// When results exist and overlay is "open", inject it into new tabs.

const API_KEY = "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";

// Track whether the overlay should be visible
let overlayOpen = false;

// Listen for messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "getApiKey") {
    sendResponse({ apiKey: API_KEY });
    return true;
  }

  if (message.action === "generateQuestions") {
    // Generate questions via OpenAI
    generateQuestions(message.data)
      .then((result) => {
        // Store results
        chrome.storage.local.set({ nieuwswijs_results: result });
        overlayOpen = true;
        chrome.storage.local.set({ nieuwswijs_overlay_open: true });
        sendResponse({ success: true, data: result });
      })
      .catch((err) => {
        sendResponse({ success: false, error: err.message });
      });
    return true; // Keep channel open for async response
  }

  if (message.action === "closeOverlay") {
    overlayOpen = false;
    chrome.storage.local.set({ nieuwswijs_overlay_open: false });
    sendResponse({ ok: true });
    return true;
  }

  if (message.action === "getOverlayState") {
    chrome.storage.local.get(["nieuwswijs_overlay_open", "nieuwswijs_results"], (data) => {
      sendResponse({
        open: data.nieuwswijs_overlay_open || false,
        results: data.nieuwswijs_results || null,
      });
    });
    return true;
  }

  if (message.action === "extractContent") {
    // Forward to active tab's content script
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]?.id) {
        chrome.tabs.sendMessage(tabs[0].id, { action: "getPageContent" }, (response) => {
          sendResponse(response);
        });
      }
    });
    return true;
  }
});

// When user clicks the extension icon, toggle the in-page panel
chrome.action.onClicked.addListener(async (tab) => {
  if (!tab?.id) return;
  try {
    // Try sending a message to toggle the panel
    await chrome.tabs.sendMessage(tab.id, { action: "togglePanel" });
  } catch (e) {
    // Content script not loaded yet — inject it first
    try {
      await chrome.scripting.executeScript({ target: { tabId: tab.id }, files: ["content.js"] });
      await chrome.scripting.insertCSS({ target: { tabId: tab.id }, files: ["overlay.css"] });
      setTimeout(() => {
        chrome.tabs.sendMessage(tab.id, { action: "togglePanel" }).catch(() => {});
      }, 200);
    } catch (err) {
      // Can't inject into chrome:// pages etc.
    }
  }
});

// When user switches tabs, inject overlay if it was open
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  const data = await chrome.storage.local.get(["nieuwswijs_overlay_open", "nieuwswijs_results"]);
  if (data.nieuwswijs_overlay_open && data.nieuwswijs_results) {
    // Small delay to let the page load
    setTimeout(() => {
      chrome.tabs.sendMessage(activeInfo.tabId, {
        action: "showOverlay",
        results: data.nieuwswijs_results,
      }).catch(() => {
        // Content script may not be loaded yet on this tab, try injecting
        injectOverlay(activeInfo.tabId, data.nieuwswijs_results);
      });
    }, 300);
  }
});

// Also inject on tab updates (navigation within a tab)
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === "complete") {
    chrome.storage.local.get(["nieuwswijs_overlay_open", "nieuwswijs_results"], (data) => {
      if (data.nieuwswijs_overlay_open && data.nieuwswijs_results) {
        setTimeout(() => {
          chrome.tabs.sendMessage(tabId, {
            action: "showOverlay",
            results: data.nieuwswijs_results,
          }).catch(() => {});
        }, 500);
      }
    });
  }
});

async function injectOverlay(tabId, results) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ["content.js"],
    });
    await chrome.scripting.insertCSS({
      target: { tabId },
      files: ["overlay.css"],
    });
    setTimeout(() => {
      chrome.tabs.sendMessage(tabId, {
        action: "showOverlay",
        results,
      }).catch(() => {});
    }, 200);
  } catch (e) {
    // Can't inject into chrome:// pages etc.
  }
}

async function generateQuestions(data) {
  const { articleTitle, articleContent, articleUrl, subject, domain, subdomain, level, klas, vakNaam, domainNaam, subdomainNaam } = data;

  const subjectInfo = `${vakNaam} (${level})`;
  const domainInfo = `${domain}`;
  const subdomainInfo = `${subdomain}`;
  const klasNr = parseInt(klas) || 4;

  // Determine difficulty description based on klas
  let klasBeschrijving = "";
  if (klasNr <= 2) {
    klasBeschrijving = `Dit is voor klas ${klasNr} (onderbouw). Houd de vragen EENVOUDIG:
- Gebruik eenvoudige taal en korte zinnen
- Toets basiskennis en begrip van kernconcepten
- Vermijd complexe analyses of abstracte redeneringen
- Multiple choice met duidelijke foute afleiders
- Open vragen die in 2-3 zinnen te beantwoorden zijn`;
  } else if (klasNr === 3) {
    klasBeschrijving = `Dit is voor klas 3. De vragen mogen iets uitdagender zijn dan onderbouw:
- Combineer basiskennis met eenvoudige toepassingsvragen
- Leerlingen mogen verbanden leggen maar hoeven nog niet diep te analyseren
- Gebruik vakbegrippen maar leg ze niet te abstract uit`;
  } else if (klasNr === 4) {
    klasBeschrijving = `Dit is voor klas 4 (bovenbouw, start examenstof). Vragen op gemiddeld examenniveau:
- Toets toepassing van concepten op nieuwe situaties
- Leerlingen moeten verbanden kunnen leggen
- Mix van kennisvragen en toepassingsvragen`;
  } else if (klasNr >= 5) {
    klasBeschrijving = `Dit is voor klas ${klasNr} (examenjaar${klasNr === 6 ? " VWO" : ""}). Vragen op VOLWAARDIG examenniveau:
- Analyseer- en evaluatievragen
- Complexe toepassingen en verbanden
- Verwacht dat leerlingen meerdere concepten kunnen combineren
- Niveau vergelijkbaar met centraal examen`;
  }

  const systemPrompt = `Je bent een ervaren ${vakNaam}-docent op het Nederlandse ${level}. Je taak is om vragen te maken die een nieuwsartikel SPECIFIEK koppelen aan de examenstof van ${vakNaam}.

KLASNIVEAU: Klas ${klasNr} ${level}
${klasBeschrijving}

BELANGRIJK - VAKSPECIFIEKE VRAGEN:
- Je maakt ALLEEN vragen vanuit het perspectief van het vak ${vakNaam}.
- Het gaat om het domein "${domainInfo}" en specifiek subdomein "${subdomainInfo}".
- De vragen moeten vakinhoudelijke concepten, theorieën en modellen uit dit subdomein toetsen.
- Gebruik het nieuwsartikel als CONTEXT/AANLEIDING, maar de kern van elke vraag moet over de EXAMENSTOF van ${vakNaam} gaan.
- Dit zijn GEEN begrijpend-lezen-vragen of Nederlands-vragen. Het gaat om de inhoud van het vak ${vakNaam}.

${vakNaam === "Economie" ? "Denk aan: economische modellen, grafieken, berekeningen, economische begrippen, marktanalyse, etc." : ""}
${vakNaam === "Aardrijkskunde" ? "Denk aan: ruimtelijke processen, kaartvaardigheden, klimaat, demografie, globalisering, etc." : ""}
${vakNaam === "Geschiedenis" ? "Denk aan: historische contexten, tijdvakken, oorzaak-gevolg, historisch redeneren, bronnenanalyse, etc." : ""}
${vakNaam === "Maatschappijleer" ? "Denk aan: politieke processen, rechtsstaat, pluriforme samenleving, parlementaire democratie, etc." : ""}
${vakNaam === "Scheikunde" ? "Denk aan: chemische reacties, molecuulstructuren, berekeningen, stofeigenschappen, etc." : ""}
${vakNaam === "Natuurkunde" ? "Denk aan: natuurkundige wetten, formules, berekeningen, experimenten, energieomzetting, etc." : ""}
${vakNaam === "Wiskunde" ? "Denk aan: wiskundige modellen, berekeningen, grafieken, statistiek, kansrekening, etc." : ""}
${vakNaam === "Nederlands" ? "Denk aan: tekstanalyse, argumentatie, taalkundige begrippen, literaire analyse, etc." : ""}
${vakNaam === "Engels" ? "Denk aan: reading comprehension, vocabulary in context, text structure, summary skills, etc." : ""}
${vakNaam === "Biologie" ? "Denk aan: cellen, stofwisseling, DNA, erfelijkheid, ecologie, evolutie, biodiversiteit, etc." : ""}

VOORBEELD van het verschil:
❌ FOUT (te algemeen/begrijpend lezen): "Wat is de hoofdgedachte van het artikel?"
❌ FOUT (te algemeen): "Noem twee argumenten uit de tekst."
✅ GOED (vakspecifiek ${vakNaam}): Een vraag die een concept uit subdomein ${subdomainInfo} toepast op de situatie beschreven in het artikel.

Genereer 5 vragen in JSON formaat:
{
  "questions": [
    {
      "questionText": "...",
      "questionType": "OPEN" of "MULTIPLE_CHOICE",
      "options": ["A) ...", "B) ...", "C) ...", "D) ..."] of null voor open vragen,
      "correctAnswer": "...",
      "explanation": "Uitleg met verwijzing naar zowel het artikel als de examenstof van ${vakNaam} subdomein ${subdomainInfo}"
    }
  ]
}

Regels:
- Maak een mix van open vragen (2-3) en multiple choice (2-3)
- Elke vraag MOET een specifiek concept/theorie/model uit subdomein ${subdomainInfo} toetsen
- Het artikel is de aanleiding, het subdomein de inhoudelijke kern
- Niveau moet passen bij klas ${klasNr} ${level}
- Gebruik vakspecifieke terminologie van ${vakNaam}`;

  const userPrompt = `Maak 5 vakspecifieke ${vakNaam}-vragen voor klas ${klasNr} ${level}. Gebruik het onderstaande nieuwsartikel als context, maar toets de examenstof van subdomein ${subdomainInfo}.

VAK: ${subjectInfo}
KLAS: ${klasNr}
DOMEIN: ${domainInfo}
SUBDOMEIN: ${subdomainInfo}

ARTIKEL:
Titel: ${articleTitle}
URL: ${articleUrl}
Inhoud: ${articleContent || "(geen inhoud beschikbaar - baseer vragen op de titel en het onderwerp in combinatie met de examenstof)"}`;

  const response = await fetch("https://api.openai.com/v1/chat/completions", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      Authorization: `Bearer ${API_KEY}`,
    },
    body: JSON.stringify({
      model: "gpt-4o-mini",
      temperature: 0.7,
      response_format: { type: "json_object" },
      messages: [
        { role: "system", content: systemPrompt },
        { role: "user", content: userPrompt },
      ],
    }),
  });

  if (!response.ok) {
    const errBody = await response.text();
    throw new Error(`OpenAI API fout (${response.status}): ${errBody}`);
  }

  const result = await response.json();
  const content = result.choices?.[0]?.message?.content;
  if (!content) throw new Error("Geen response van OpenAI");

  const parsed = JSON.parse(content);
  const questions = parsed.questions || [];

  if (questions.length === 0) {
    throw new Error("Geen vragen gegenereerd");
  }

  return {
    questions,
    articleTitle,
    articleUrl,
    subject: subjectInfo,
    klas: klasNr,
    domain: domainInfo,
    subdomain: subdomainInfo,
    generatedAt: new Date().toISOString(),
  };
}
