// ==============================================================
// Nieuwswijs Chrome Extension - Content Script
// ==============================================================
// Runs on every page. Injects a draggable panel for selecting
// niveau/klas/vak/domein/subdomein + generating & displaying questions.

let panelElement = null;
let overlayElement = null;
let currentResults = null;
let CURRICULUM = null;

// ---- Load curriculum data ----
function loadCurriculum() {
  if (CURRICULUM) return Promise.resolve();
  return fetch(chrome.runtime.getURL("curriculum.json"))
    .then(r => r.json())
    .then(data => { CURRICULUM = data; });
}

// ---- Page content extraction ----
function getPageContent() {
  const title = document.title || "";
  const url = window.location.href;
  const host = window.location.hostname;
  const selectors = ["article", '[role="article"]', ".article-body", ".article__body", ".article-content", ".post-content", ".entry-content", "main"];
  let content = "";
  for (const sel of selectors) {
    const el = document.querySelector(sel);
    if (el && el.innerText.length > 200) { content = el.innerText; break; }
  }
  if (!content || content.length < 200) {
    const meta = document.querySelector('meta[name="description"]');
    const metaDesc = meta ? meta.getAttribute("content") || "" : "";
    const paragraphs = Array.from(document.querySelectorAll("p")).map(p => p.innerText.trim()).filter(t => t.length > 40);
    content = metaDesc + "\n\n" + paragraphs.join("\n\n");
  }
  if (content.length > 4000) content = content.substring(0, 4000) + "...";
  return { title, url, host, content };
}

// ---- Klas ranges per niveau ----
const KLAS_RANGES = { VMBO: [1,2,3,4], HAVO: [1,2,3,4,5], VWO: [1,2,3,4,5,6] };

// ==============================
// SELECTION PANEL (draggable)
// ==============================
async function createPanel() {
  if (panelElement) { panelElement.style.display = "flex"; return; }
  await loadCurriculum();

  const panel = document.createElement("div");
  panel.id = "nieuwswijs-panel";
  panel.innerHTML = `
    <div id="nw-panel-header">
      <span class="nw-panel-title">Nieuwswijs</span>
      <button id="nw-panel-close" title="Sluiten">&times;</button>
    </div>
    <div id="nw-panel-body">
      <div id="nw-page-info" style="display:none;">
        <div id="nw-page-title"></div>
        <div id="nw-page-source"></div>
      </div>
      <div class="nw-field">
        <label>Niveau</label>
        <select id="nw-niveau"><option value="">Kies niveau...</option>
          <option value="VMBO">VMBO</option><option value="HAVO">HAVO</option><option value="VWO">VWO</option>
        </select>
      </div>
      <div class="nw-field">
        <label>Klas</label>
        <select id="nw-klas" disabled><option value="">Kies eerst niveau...</option></select>
      </div>
      <div class="nw-field">
        <label>Vak</label>
        <select id="nw-vak" disabled><option value="">Kies eerst niveau...</option></select>
      </div>
      <div class="nw-field">
        <label>Domein</label>
        <select id="nw-domein" disabled><option value="">Kies eerst vak...</option></select>
      </div>
      <div class="nw-field">
        <label>Subdomein</label>
        <select id="nw-subdomein" disabled><option value="">Kies eerst domein...</option></select>
      </div>
      <button id="nw-btn-generate" disabled>Genereer</button>
      <div id="nw-status"></div>
    </div>
  `;
  document.body.appendChild(panel);
  panelElement = panel;

  // Make draggable
  makeDraggable(panel, panel.querySelector("#nw-panel-header"));

  // Close
  panel.querySelector("#nw-panel-close").addEventListener("click", () => { panel.style.display = "none"; });

  // Show page info
  const pc = getPageContent();
  if (pc.title) {
    panel.querySelector("#nw-page-info").style.display = "block";
    panel.querySelector("#nw-page-title").textContent = pc.title;
    panel.querySelector("#nw-page-source").textContent = pc.host;
  }

  // Dropdown logic
  const niveau = panel.querySelector("#nw-niveau");
  const klas = panel.querySelector("#nw-klas");
  const vak = panel.querySelector("#nw-vak");
  const domein = panel.querySelector("#nw-domein");
  const subdomein = panel.querySelector("#nw-subdomein");
  const btn = panel.querySelector("#nw-btn-generate");
  const status = panel.querySelector("#nw-status");

  function resetFrom(field) {
    if (field === "klas" || field === "all") { klas.innerHTML = '<option value="">Kies klas...</option>'; klas.disabled = true; }
    if (field === "vak" || field === "all") { vak.innerHTML = '<option value="">Kies vak...</option>'; vak.disabled = true; }
    if (field === "domein" || field === "all" || field === "vak") { domein.innerHTML = '<option value="">Kies domein...</option>'; domein.disabled = true; }
    if (field === "subdomein" || field === "all" || field === "vak" || field === "domein") { subdomein.innerHTML = '<option value="">Kies subdomein...</option>'; subdomein.disabled = true; }
    btn.disabled = true;
  }

  niveau.addEventListener("change", () => {
    resetFrom("all");
    if (!niveau.value) return;
    // Klas
    const klassen = KLAS_RANGES[niveau.value] || [];
    klas.innerHTML = '<option value="">Kies klas...</option>';
    klassen.forEach(k => { const o = document.createElement("option"); o.value = k; o.textContent = `Klas ${k}`; klas.appendChild(o); });
    klas.disabled = false;
    // Vak
    const subjects = CURRICULUM.filter(s => s.level === niveau.value);
    vak.innerHTML = '<option value="">Kies vak...</option>';
    subjects.forEach(s => { const o = document.createElement("option"); o.value = s.slug; o.textContent = s.name; vak.appendChild(o); });
    vak.disabled = false;
  });

  klas.addEventListener("change", () => { checkReady(); });

  vak.addEventListener("change", () => {
    resetFrom("domein");
    if (!vak.value) return;
    const subject = CURRICULUM.find(s => s.level === niveau.value && s.slug === vak.value);
    if (!subject) return;
    domein.innerHTML = '<option value="">Kies domein...</option>';
    subject.domains.forEach(d => { const o = document.createElement("option"); o.value = d.code; o.textContent = `${d.code} - ${d.name}`; domein.appendChild(o); });
    domein.disabled = false;
  });

  domein.addEventListener("change", () => {
    resetFrom("subdomein");
    if (!domein.value) return;
    const subject = CURRICULUM.find(s => s.level === niveau.value && s.slug === vak.value);
    const dom = subject?.domains.find(d => d.code === domein.value);
    if (!dom) return;
    subdomein.innerHTML = '<option value="">Kies subdomein...</option>';
    dom.subdomains.forEach(sd => { const o = document.createElement("option"); o.value = sd.code; o.textContent = `${sd.code} - ${sd.name}`; subdomein.appendChild(o); });
    subdomein.disabled = false;
  });

  subdomein.addEventListener("change", () => { checkReady(); });

  function checkReady() {
    btn.disabled = !(niveau.value && klas.value && vak.value && domein.value && subdomein.value);
  }

  // Generate
  btn.addEventListener("click", async () => {
    const subject = CURRICULUM.find(s => s.level === niveau.value && s.slug === vak.value);
    const dom = subject?.domains.find(d => d.code === domein.value);
    const sub = dom?.subdomains.find(sd => sd.code === subdomein.value);
    if (!subject || !dom || !sub) return;

    btn.disabled = true;
    btn.textContent = "Genereren...";
    status.textContent = "Vragen worden gegenereerd...";
    status.className = "nw-status-loading";

    // Show loading overlay
    showLoading();

    const pc = getPageContent();
    try {
      const response = await chrome.runtime.sendMessage({
        action: "generateQuestions",
        data: {
          articleTitle: pc.title || "Onbekend artikel",
          articleContent: pc.content || "",
          articleUrl: pc.url || "",
          level: niveau.value,
          klas: klas.value,
          vakNaam: subject.name,
          subject: `${subject.name} (${niveau.value})`,
          domain: `${dom.code} - ${dom.name}`,
          subdomain: `${sub.code} - ${sub.name}`,
          domainNaam: dom.name,
          subdomainNaam: sub.name,
        },
      });

      if (response.success) {
        status.textContent = `${response.data.questions.length} vragen gegenereerd!`;
        status.className = "";
        renderResults(response.data);
        // Hide panel, show results
        panel.style.display = "none";
      } else {
        throw new Error(response.error || "Genereren mislukt");
      }
    } catch (err) {
      status.textContent = err.message || "Er ging iets mis.";
      status.className = "nw-status-error";
      hideOverlay();
    } finally {
      btn.disabled = false;
      btn.textContent = "Genereer";
    }
  });

  // Restore selections
  chrome.storage.local.get("nieuwswijs_selections", (data) => {
    const sel = data.nieuwswijs_selections;
    if (!sel) return;
    if (sel.niveau) { niveau.value = sel.niveau; niveau.dispatchEvent(new Event("change")); }
    setTimeout(() => {
      if (sel.klas) { klas.value = sel.klas; klas.dispatchEvent(new Event("change")); }
      if (sel.vak) { vak.value = sel.vak; vak.dispatchEvent(new Event("change")); }
      setTimeout(() => {
        if (sel.domein) { domein.value = sel.domein; domein.dispatchEvent(new Event("change")); }
        setTimeout(() => {
          if (sel.subdomein) { subdomein.value = sel.subdomein; subdomein.dispatchEvent(new Event("change")); }
        }, 10);
      }, 10);
    }, 10);
  });

  // Save selections on change
  [niveau, klas, vak, domein, subdomein].forEach(el => {
    el.addEventListener("change", () => {
      chrome.storage.local.set({ nieuwswijs_selections: {
        niveau: niveau.value, klas: klas.value, vak: vak.value, domein: domein.value, subdomein: subdomein.value
      }});
      checkReady();
    });
  });
}

// ==============================
// RESULTS OVERLAY
// ==============================
function createOverlay() {
  if (overlayElement) return overlayElement;
  const overlay = document.createElement("div");
  overlay.id = "nieuwswijs-overlay";
  overlay.classList.add("nw-hidden");
  overlay.innerHTML = `
    <div id="nieuwswijs-topbar">
      <span class="nw-title">Nieuwswijs</span>
      <div id="nw-topbar-buttons">
        <button id="nw-btn-back" title="Terug naar selectie">←</button>
        <button id="nw-btn-export" title="Exporteer naar Word" style="display:none;">Exporteer</button>
        <button id="nieuwswijs-close-btn" title="Sluiten">&times;</button>
      </div>
    </div>
    <div id="nieuwswijs-meta" style="display:none;">
      <div class="nw-meta-subject" id="nw-meta-subject"></div>
      <div class="nw-meta-article" id="nw-meta-article"></div>
    </div>
    <div id="nieuwswijs-content">
      <div id="nieuwswijs-loading" style="display:none;">
        <div class="nw-spinner"></div>
        <div class="nw-loading-text">Vragen genereren...</div>
      </div>
      <div id="nieuwswijs-questions"></div>
    </div>
    <div id="nieuwswijs-footer" style="display:none;">
      <button class="nw-btn-export-footer" id="nw-btn-export-footer">📄 Exporteer naar Word</button>
      <div class="nw-powered">Gegenereerd door Nieuwswijs</div>
    </div>
  `;
  document.body.appendChild(overlay);
  overlayElement = overlay;

  // Close
  overlay.querySelector("#nieuwswijs-close-btn").addEventListener("click", () => {
    hideOverlay();
    chrome.runtime.sendMessage({ action: "closeOverlay" });
  });

  // Back to selection panel
  overlay.querySelector("#nw-btn-back").addEventListener("click", () => {
    hideOverlay();
    chrome.runtime.sendMessage({ action: "closeOverlay" });
    createPanel();
  });

  // Export buttons
  overlay.querySelector("#nw-btn-export").addEventListener("click", exportToWord);
  overlay.querySelector("#nw-btn-export-footer").addEventListener("click", exportToWord);

  // Question interaction
  overlay.querySelector("#nieuwswijs-questions").addEventListener("click", handleQuestionClick);

  return overlay;
}

function showOverlay() {
  const overlay = createOverlay();
  requestAnimationFrame(() => overlay.classList.remove("nw-hidden"));
}

function hideOverlay() {
  if (overlayElement) overlayElement.classList.add("nw-hidden");
}

function showLoading() {
  const overlay = createOverlay();
  showOverlay();
  const loading = overlay.querySelector("#nieuwswijs-loading");
  const questions = overlay.querySelector("#nieuwswijs-questions");
  const footer = overlay.querySelector("#nieuwswijs-footer");
  const meta = overlay.querySelector("#nieuwswijs-meta");
  const exportBtn = overlay.querySelector("#nw-btn-export");
  if (loading) loading.style.display = "block";
  if (questions) questions.innerHTML = "";
  if (footer) footer.style.display = "none";
  if (meta) meta.style.display = "none";
  if (exportBtn) exportBtn.style.display = "none";
}

function renderResults(results) {
  if (!results || !results.questions) return;
  currentResults = results;

  const overlay = createOverlay();
  showOverlay();

  const loading = overlay.querySelector("#nieuwswijs-loading");
  const questionsEl = overlay.querySelector("#nieuwswijs-questions");
  const footer = overlay.querySelector("#nieuwswijs-footer");
  const meta = overlay.querySelector("#nieuwswijs-meta");
  const metaSubject = overlay.querySelector("#nw-meta-subject");
  const metaArticle = overlay.querySelector("#nw-meta-article");
  const exportBtn = overlay.querySelector("#nw-btn-export");

  if (loading) loading.style.display = "none";
  if (footer) footer.style.display = "block";
  if (meta) meta.style.display = "block";
  if (exportBtn) exportBtn.style.display = "inline-block";

  if (metaSubject) metaSubject.textContent = `${results.subject}${results.klas ? ` · Klas ${results.klas}` : ""} · ${results.domain} · ${results.subdomain}`;
  if (metaArticle) {
    if (results.articleUrl) {
      metaArticle.innerHTML = `<a href="${esc(results.articleUrl)}" target="_blank">${esc(results.articleTitle)}</a>`;
    } else {
      metaArticle.textContent = results.articleTitle;
    }
  }

  if (!questionsEl) return;
  questionsEl.innerHTML = "";

  results.questions.forEach((q, i) => {
    const card = document.createElement("div");
    card.className = "nw-question-card";
    const isMC = q.questionType === "MULTIPLE_CHOICE";
    const badgeClass = isMC ? "nw-mc" : "nw-open";
    const typeLabel = isMC ? "Multiple choice" : "Open vraag";

    let html = `
      <span class="nw-question-badge ${badgeClass}">Vraag ${i + 1}</span>
      <span class="nw-question-type-label">${typeLabel}</span>
      <p class="nw-question-text">${esc(q.questionText)}</p>
    `;

    if (isMC && q.options) {
      html += '<div class="nw-options">';
      q.options.forEach((opt, oi) => {
        html += `<div class="nw-option" data-q="${i}" data-o="${oi}">${esc(opt)}</div>`;
      });
      html += "</div>";
    } else {
      html += `<textarea class="nw-open-answer" placeholder="Typ hier je antwoord..."></textarea>`;
    }

    html += `
      <button class="nw-btn-show-answer" data-q="${i}">Toon antwoord</button>
      <div class="nw-answer-section" id="nw-answer-${i}">
        <div class="nw-answer-label">Antwoord</div>
        <p class="nw-answer-text">${esc(q.correctAnswer)}</p>
        ${q.explanation ? `<div class="nw-explanation-label">Uitleg</div><p class="nw-explanation-text">${esc(q.explanation)}</p>` : ""}
      </div>
    `;

    card.innerHTML = html;
    questionsEl.appendChild(card);
  });
}

function handleQuestionClick(e) {
  if (!currentResults) return;
  const option = e.target.closest(".nw-option");
  if (option) {
    const qi = parseInt(option.dataset.q);
    const q = currentResults.questions[qi];
    if (!q) return;
    if (option.classList.contains("nw-correct") || option.classList.contains("nw-wrong")) return;

    const clickedLetter = option.textContent.trim().match(/^([A-D])\)/)?.[1];
    const correctLetter = q.correctAnswer.trim().match(/^([A-D])\)/)?.[1];
    const isCorrect = correctLetter && clickedLetter ? correctLetter === clickedLetter : option.textContent.trim() === q.correctAnswer.trim();
    const allOpts = overlayElement.querySelectorAll(`.nw-option[data-q="${qi}"]`);

    if (isCorrect) {
      option.classList.add("nw-correct");
    } else {
      option.classList.add("nw-wrong");
      allOpts.forEach(opt => {
        const l = opt.textContent.trim().match(/^([A-D])\)/)?.[1];
        if (correctLetter && l === correctLetter) opt.classList.add("nw-correct");
      });
    }
    const ans = overlayElement.querySelector(`#nw-answer-${qi}`);
    if (ans) ans.classList.add("nw-visible");
    return;
  }

  const btn = e.target.closest(".nw-btn-show-answer");
  if (btn) {
    const qi = btn.dataset.q;
    const ans = overlayElement.querySelector(`#nw-answer-${qi}`);
    if (ans) ans.classList.add("nw-visible");
    btn.style.display = "none";
  }
}

// ==============================
// EXPORT TO WORD (.docx)
// ==============================

// --- Minimal ZIP creator (no libraries) ---
function _crc32Table() {
  const t = new Uint32Array(256);
  for (let i = 0; i < 256; i++) {
    let c = i;
    for (let j = 0; j < 8; j++) c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
    t[i] = c;
  }
  return t;
}
const _CRC_T = _crc32Table();
function _crc32(data) {
  const b = typeof data === "string" ? new TextEncoder().encode(data) : data;
  let c = 0xFFFFFFFF;
  for (let i = 0; i < b.length; i++) c = _CRC_T[(c ^ b[i]) & 0xFF] ^ (c >>> 8);
  return (c ^ 0xFFFFFFFF) >>> 0;
}
function _createZip(files) {
  const enc = new TextEncoder();
  const records = files.map(f => {
    const p = enc.encode(f.path);
    const d = typeof f.data === "string" ? enc.encode(f.data) : f.data;
    return { p, d, crc: _crc32(d) };
  });
  let lhSize = 0;
  records.forEach(r => { lhSize += 30 + r.p.length + r.d.length; });
  let cdSize = 0;
  records.forEach(r => { cdSize += 46 + r.p.length; });
  const buf = new ArrayBuffer(lhSize + cdSize + 22);
  const v = new DataView(buf);
  const u = new Uint8Array(buf);
  let o = 0;
  const offsets = [];
  records.forEach(r => {
    offsets.push(o);
    v.setUint32(o, 0x04034b50, true); o += 4;
    v.setUint16(o, 20, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint32(o, r.crc, true); o += 4;
    v.setUint32(o, r.d.length, true); o += 4;
    v.setUint32(o, r.d.length, true); o += 4;
    v.setUint16(o, r.p.length, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    u.set(r.p, o); o += r.p.length;
    u.set(r.d, o); o += r.d.length;
  });
  const cdOff = o;
  records.forEach((r, i) => {
    v.setUint32(o, 0x02014b50, true); o += 4;
    v.setUint16(o, 20, true); o += 2;
    v.setUint16(o, 20, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint32(o, r.crc, true); o += 4;
    v.setUint32(o, r.d.length, true); o += 4;
    v.setUint32(o, r.d.length, true); o += 4;
    v.setUint16(o, r.p.length, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint16(o, 0, true); o += 2;
    v.setUint32(o, 0, true); o += 4;
    v.setUint32(o, offsets[i], true); o += 4;
    u.set(r.p, o); o += r.p.length;
  });
  v.setUint32(o, 0x06054b50, true); o += 4;
  v.setUint16(o, 0, true); o += 2;
  v.setUint16(o, 0, true); o += 2;
  v.setUint16(o, records.length, true); o += 2;
  v.setUint16(o, records.length, true); o += 2;
  v.setUint32(o, cdSize, true); o += 4;
  v.setUint32(o, cdOff, true); o += 4;
  v.setUint16(o, 0, true);
  return new Blob([buf], { type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document" });
}

// --- XML escape for docx ---
function xmlEsc(s) {
  if (!s) return "";
  return s.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;");
}

// --- Build the .docx and download ---
function exportToWord() {
  if (!currentResults) return;
  const r = currentResults;

  // Build Word XML paragraphs for questions + answers only
  let body = "";

  // Title
  body += `<w:p><w:pPr><w:spacing w:after="120"/></w:pPr><w:r><w:rPr><w:b/><w:sz w:val="36"/><w:color w:val="1E40AF"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>Nieuwswijs - Vragen</w:t></w:r></w:p>`;

  // Subtitle: vak, klas, domein
  body += `<w:p><w:pPr><w:spacing w:after="200"/></w:pPr><w:r><w:rPr><w:sz w:val="22"/><w:color w:val="666666"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>${xmlEsc(r.subject)}${r.klas ? ` \u00B7 Klas ${r.klas}` : ""} \u00B7 ${xmlEsc(r.domain)} \u00B7 ${xmlEsc(r.subdomain)}</w:t></w:r></w:p>`;

  // Horizontal line
  body += `<w:p><w:pPr><w:pBdr><w:bottom w:val="single" w:sz="6" w:space="1" w:color="CCCCCC"/></w:pBdr><w:spacing w:after="200"/></w:pPr></w:p>`;

  r.questions.forEach((q, i) => {
    const type = q.questionType === "MULTIPLE_CHOICE" ? "Multiple choice" : "Open vraag";

    // Question heading: "Vraag 1 (Multiple choice)"
    body += `<w:p><w:pPr><w:spacing w:before="240" w:after="80"/></w:pPr><w:r><w:rPr><w:b/><w:sz w:val="24"/><w:color w:val="1E40AF"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>Vraag ${i + 1} (${type})</w:t></w:r></w:p>`;

    // Question text
    body += `<w:p><w:pPr><w:spacing w:after="80"/></w:pPr><w:r><w:rPr><w:sz w:val="22"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>${xmlEsc(q.questionText)}</w:t></w:r></w:p>`;

    // MC options
    if (q.options) {
      q.options.forEach(opt => {
        body += `<w:p><w:pPr><w:ind w:left="360"/><w:spacing w:after="40"/></w:pPr><w:r><w:rPr><w:sz w:val="22"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>${xmlEsc(opt)}</w:t></w:r></w:p>`;
      });
    }

    // "Antwoord" label
    body += `<w:p><w:pPr><w:spacing w:before="120" w:after="40"/></w:pPr><w:r><w:rPr><w:b/><w:sz w:val="18"/><w:color w:val="16A34A"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>ANTWOORD</w:t></w:r></w:p>`;

    // Answer text
    body += `<w:p><w:pPr><w:spacing w:after="40"/></w:pPr><w:r><w:rPr><w:b/><w:sz w:val="22"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>${xmlEsc(q.correctAnswer)}</w:t></w:r></w:p>`;

    // Explanation
    if (q.explanation) {
      body += `<w:p><w:pPr><w:spacing w:before="40" w:after="80"/></w:pPr><w:r><w:rPr><w:sz w:val="20"/><w:color w:val="666666"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>${xmlEsc(q.explanation)}</w:t></w:r></w:p>`;
    }

    // Separator line between questions
    if (i < r.questions.length - 1) {
      body += `<w:p><w:pPr><w:pBdr><w:bottom w:val="single" w:sz="4" w:space="1" w:color="DDDDDD"/></w:pBdr><w:spacing w:after="120"/></w:pPr></w:p>`;
    }
  });

  // Footer
  body += `<w:p><w:pPr><w:spacing w:before="360"/></w:pPr><w:r><w:rPr><w:sz w:val="16"/><w:color w:val="AAAAAA"/><w:rFonts w:ascii="Calibri" w:hAnsi="Calibri"/></w:rPr><w:t>Gegenereerd door Nieuwswijs</w:t></w:r></w:p>`;

  // Assemble the docx XML files
  const contentTypes = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?><Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types"><Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/><Default Extension="xml" ContentType="application/xml"/><Override PartName="/word/document.xml" ContentType="application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml"/></Types>`;

  const rels = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?><Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships"><Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="word/document.xml"/></Relationships>`;

  const docXml = `<?xml version="1.0" encoding="UTF-8" standalone="yes"?><w:document xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main"><w:body>${body}</w:body></w:document>`;

  const blob = _createZip([
    { path: "[Content_Types].xml", data: contentTypes },
    { path: "_rels/.rels", data: rels },
    { path: "word/document.xml", data: docXml },
  ]);

  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  const vakNaam = r.subject.split(" (")[0] || "Nieuwswijs";
  a.download = `Nieuwswijs_${vakNaam}_Klas${r.klas || ""}_${new Date().toISOString().slice(0,10)}.docx`;
  document.body.appendChild(a);
  a.click();
  document.body.removeChild(a);
  URL.revokeObjectURL(url);
}

// ==============================
// DRAG HELPER
// ==============================
function makeDraggable(element, handle) {
  let isDragging = false;
  let startX, startY, startLeft, startTop;

  handle.style.cursor = "grab";

  handle.addEventListener("mousedown", (e) => {
    if (e.target.tagName === "BUTTON" || e.target.tagName === "SELECT") return;
    isDragging = true;
    handle.style.cursor = "grabbing";
    const rect = element.getBoundingClientRect();
    startX = e.clientX;
    startY = e.clientY;
    startLeft = rect.left;
    startTop = rect.top;
    e.preventDefault();
  });

  document.addEventListener("mousemove", (e) => {
    if (!isDragging) return;
    const dx = e.clientX - startX;
    const dy = e.clientY - startY;
    element.style.left = (startLeft + dx) + "px";
    element.style.top = (startTop + dy) + "px";
    element.style.right = "auto";
    element.style.bottom = "auto";
  });

  document.addEventListener("mouseup", () => {
    if (isDragging) {
      isDragging = false;
      handle.style.cursor = "grab";
    }
  });
}

// ==============================
// UTILITIES
// ==============================
function esc(text) {
  if (!text) return "";
  const d = document.createElement("div");
  d.textContent = text;
  return d.innerHTML;
}

// ==============================
// MESSAGE LISTENER
// ==============================
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "getPageContent") {
    sendResponse(getPageContent());
    return true;
  }
  if (message.action === "togglePanel") {
    if (panelElement && panelElement.style.display !== "none") {
      panelElement.style.display = "none";
    } else {
      createPanel();
    }
    sendResponse({ ok: true });
    return true;
  }
  if (message.action === "showLoading") {
    showLoading();
    sendResponse({ ok: true });
    return true;
  }
  if (message.action === "showOverlay") {
    renderResults(message.results);
    sendResponse({ ok: true });
    return true;
  }
  if (message.action === "hideOverlay") {
    hideOverlay();
    sendResponse({ ok: true });
    return true;
  }
});

// ---- On load: check if overlay should be shown ----
chrome.runtime.sendMessage({ action: "getOverlayState" }, (response) => {
  if (chrome.runtime.lastError) return;
  if (response?.open && response?.results) {
    renderResults(response.results);
  }
});
