// ==============================================================
// Nieuwswijs Chrome Extension - Popup Logic
// ==============================================================

const niveauSelect = document.getElementById("niveau");
const klasSelect = document.getElementById("klas");
const vakSelect = document.getElementById("vak");
const domeinSelect = document.getElementById("domein");
const subdomeinSelect = document.getElementById("subdomein");
const btnGenerate = document.getElementById("btnGenerate");
const statusEl = document.getElementById("status");
const pageInfoEl = document.getElementById("pageInfo");
const pageTitleEl = document.getElementById("pageTitle");
const pageSourceEl = document.getElementById("pageSource");

let pageContent = null;

const KLAS_RANGES = { VMBO: [1,2,3,4], HAVO: [1,2,3,4,5], VWO: [1,2,3,4,5,6] };

// ---- Page content extraction ----
async function extractPageContent() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab?.id) return;

    const results = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: () => {
        const title = document.title || "";
        const url = window.location.href;
        const host = window.location.hostname;
        const selectors = ["article", '[role="article"]', ".article-body", ".article__body", ".article-content", ".post-content", ".entry-content", "main"];
        let content = "";
        for (const sel of selectors) {
          const el = document.querySelector(sel);
          if (el && el.innerText.length > 200) { content = el.innerText; break; }
        }
        if (!content || content.length < 200) {
          const meta = document.querySelector('meta[name="description"]');
          const metaDesc = meta ? meta.getAttribute("content") || "" : "";
          const paragraphs = Array.from(document.querySelectorAll("p")).map((p) => p.innerText.trim()).filter((t) => t.length > 40);
          content = metaDesc + "\n\n" + paragraphs.join("\n\n");
        }
        if (content.length > 4000) content = content.substring(0, 4000) + "...";
        return { title, url, host, content };
      },
    });

    if (results?.[0]?.result) {
      pageContent = results[0].result;
      pageTitleEl.textContent = pageContent.title;
      pageSourceEl.textContent = pageContent.host;
      pageInfoEl.style.display = "block";
    }
  } catch (err) {
    console.log("Could not extract page content:", err);
  }
}

// ---- Dropdown helpers ----
function getSubjectsForLevel(level) {
  return CURRICULUM.filter((s) => s.level === level);
}
function getSubject(level, slug) {
  return CURRICULUM.find((s) => s.level === level && s.slug === slug);
}

function checkReady() {
  btnGenerate.disabled = !(niveauSelect.value && klasSelect.value && vakSelect.value && domeinSelect.value && subdomeinSelect.value);
}

// ---- Dropdown events ----
niveauSelect.addEventListener("change", () => {
  const level = niveauSelect.value;
  // Reset all downstream
  klasSelect.innerHTML = '<option value="">Kies klas...</option>';
  vakSelect.innerHTML = '<option value="">Kies vak...</option>';
  domeinSelect.innerHTML = '<option value="">Kies eerst vak...</option>';
  subdomeinSelect.innerHTML = '<option value="">Kies eerst domein...</option>';
  klasSelect.disabled = true;
  vakSelect.disabled = true;
  domeinSelect.disabled = true;
  subdomeinSelect.disabled = true;
  btnGenerate.disabled = true;

  if (!level) return;

  // Klas
  const klassen = KLAS_RANGES[level] || [];
  klassen.forEach((k) => {
    const opt = document.createElement("option");
    opt.value = k;
    opt.textContent = `Klas ${k}`;
    klasSelect.appendChild(opt);
  });
  klasSelect.disabled = false;

  // Vak
  const subjects = getSubjectsForLevel(level);
  subjects.forEach((s) => {
    const opt = document.createElement("option");
    opt.value = s.slug;
    opt.textContent = s.name;
    vakSelect.appendChild(opt);
  });
  vakSelect.disabled = false;
});

klasSelect.addEventListener("change", checkReady);

vakSelect.addEventListener("change", () => {
  domeinSelect.innerHTML = '<option value="">Kies domein...</option>';
  subdomeinSelect.innerHTML = '<option value="">Kies eerst domein...</option>';
  subdomeinSelect.disabled = true;
  btnGenerate.disabled = true;

  if (!vakSelect.value) { domeinSelect.disabled = true; return; }

  const subject = getSubject(niveauSelect.value, vakSelect.value);
  if (!subject) return;
  subject.domains.forEach((d) => {
    const opt = document.createElement("option");
    opt.value = d.code;
    opt.textContent = `${d.code} - ${d.name}`;
    domeinSelect.appendChild(opt);
  });
  domeinSelect.disabled = false;
});

domeinSelect.addEventListener("change", () => {
  subdomeinSelect.innerHTML = '<option value="">Kies subdomein...</option>';
  btnGenerate.disabled = true;

  if (!domeinSelect.value) { subdomeinSelect.disabled = true; return; }

  const subject = getSubject(niveauSelect.value, vakSelect.value);
  const domain = subject?.domains.find((d) => d.code === domeinSelect.value);
  if (!domain) return;
  domain.subdomains.forEach((sd) => {
    const opt = document.createElement("option");
    opt.value = sd.code;
    opt.textContent = `${sd.code} - ${sd.name}`;
    subdomeinSelect.appendChild(opt);
  });
  subdomeinSelect.disabled = false;
});

subdomeinSelect.addEventListener("change", checkReady);

// ---- Question generation ----
btnGenerate.addEventListener("click", async () => {
  const level = niveauSelect.value;
  const klas = klasSelect.value;
  const slug = vakSelect.value;
  const domainCode = domeinSelect.value;
  const subdomainCode = subdomeinSelect.value;

  if (!level || !klas || !slug || !domainCode || !subdomainCode) return;

  const subject = getSubject(level, slug);
  const domain = subject?.domains.find((d) => d.code === domainCode);
  const subdomain = domain?.subdomains.find((sd) => sd.code === subdomainCode);
  if (!subject || !domain || !subdomain) return;

  btnGenerate.disabled = true;
  btnGenerate.textContent = "Genereren...";
  statusEl.textContent = "Vragen worden gegenereerd...";
  statusEl.className = "status loading";

  try {
    const articleTitle = pageContent?.title || "Onbekend artikel";
    const articleContent = pageContent?.content || "";
    const articleUrl = pageContent?.url || "";

    // Show loading overlay in the page
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab?.id) {
      chrome.tabs.sendMessage(tab.id, { action: "showLoading" }).catch(() => {});
    }

    // Send to background for generation
    const response = await chrome.runtime.sendMessage({
      action: "generateQuestions",
      data: {
        articleTitle,
        articleContent,
        articleUrl,
        level,
        klas,
        vakNaam: subject.name,
        subject: `${subject.name} (${level})`,
        domain: `${domain.code} - ${domain.name}`,
        subdomain: `${subdomain.code} - ${subdomain.name}`,
        domainNaam: domain.name,
        subdomainNaam: subdomain.name,
      },
    });

    if (response.success) {
      statusEl.textContent = `${response.data.questions.length} vragen gegenereerd!`;
      statusEl.className = "status";

      // Show overlay in the current tab
      if (tab?.id) {
        chrome.tabs.sendMessage(tab.id, {
          action: "showOverlay",
          results: response.data,
        }).catch(() => {});
      }

      // Close popup after short delay
      setTimeout(() => window.close(), 500);
    } else {
      throw new Error(response.error || "Genereren mislukt");
    }
  } catch (err) {
    console.error("Generation error:", err);
    statusEl.textContent = err.message || "Er ging iets mis.";
    statusEl.className = "status error";
  } finally {
    btnGenerate.disabled = false;
    btnGenerate.textContent = "Genereer";
  }
});

// ---- Init ----
extractPageContent();

// Restore last used selections
chrome.storage.local.get("nieuwswijs_selections", (data) => {
  const sel = data.nieuwswijs_selections;
  if (sel) {
    niveauSelect.value = sel.niveau || "";
    niveauSelect.dispatchEvent(new Event("change"));
    setTimeout(() => {
      if (sel.klas) { klasSelect.value = sel.klas; klasSelect.dispatchEvent(new Event("change")); }
      if (sel.vak) { vakSelect.value = sel.vak; vakSelect.dispatchEvent(new Event("change")); }
      setTimeout(() => {
        if (sel.domein) { domeinSelect.value = sel.domein; domeinSelect.dispatchEvent(new Event("change")); }
        setTimeout(() => {
          if (sel.subdomein) { subdomeinSelect.value = sel.subdomein; subdomeinSelect.dispatchEvent(new Event("change")); }
        }, 10);
      }, 10);
    }, 10);
  }
});

// Save selections on change
[niveauSelect, klasSelect, vakSelect, domeinSelect, subdomeinSelect].forEach((el) => {
  el.addEventListener("change", () => {
    chrome.storage.local.set({
      nieuwswijs_selections: {
        niveau: niveauSelect.value,
        klas: klasSelect.value,
        vak: vakSelect.value,
        domein: domeinSelect.value,
        subdomein: subdomeinSelect.value,
      },
    });
  });
});
